<?php
require_once 'rb.php';

// --- CONFIGURATION ---
$db_file = 'mwalimu.db';
$dsn = "sqlite:$db_file";
R::setup($dsn);
R::freeze(false);

function h($str) { return htmlspecialchars($str, ENT_QUOTES, 'UTF-8'); }

$table = $_GET['table'] ?? '';
$action = $_GET['action'] ?? 'list';
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 10;
$search = trim($_GET['search'] ?? '');

$tables = R::inspect();
if ($table !== '' && !in_array($table, $tables)) die('Invalid table.');

// Pagination helper
function paginate($total, $per_page, $page, $base_url) {
    $pages = max(1, ceil($total / $per_page));
    $page = max(1, min($page, $pages));
    echo '<nav><ul class="pagination justify-content-center">';
    echo $page > 1
        ? '<li class="page-item"><a class="page-link" href="' . h($base_url) . '&page=1">First</a></li>'
        : '<li class="page-item disabled"><span class="page-link">First</span></li>';
    echo $page > 1
        ? '<li class="page-item"><a class="page-link" href="' . h($base_url) . '&page=' . ($page - 1) . '">Prev</a></li>'
        : '<li class="page-item disabled"><span class="page-link">Prev</span></li>';
    echo '<li class="page-item disabled"><span class="page-link">Page ' . $page . ' of ' . $pages . ' (Total: ' . $total . ')</span></li>';
    echo $page < $pages
        ? '<li class="page-item"><a class="page-link" href="' . h($base_url) . '&page=' . ($page + 1) . '">Next</a></li>'
        : '<li class="page-item disabled"><span class="page-link">Next</span></li>';
    echo $page < $pages
        ? '<li class="page-item"><a class="page-link" href="' . h($base_url) . '&page=' . $pages . '">Last</a></li>'
        : '<li class="page-item disabled"><span class="page-link">Last</span></li>';
    echo '</ul></nav>';
}

// Search filter
$where = '';
$params = [];
if ($search !== '' && $table !== '') {
    $columns = R::getColumns($table);
    $textColumns = [];
    foreach ($columns as $col => $type) {
        if (stripos($type, 'char') !== false || stripos($type, 'text') !== false) {
            $textColumns[] = $col;
        }
    }
    if ($textColumns) {
        $likeClauses = [];
        foreach ($textColumns as $col) {
            $likeClauses[] = "$col LIKE ?";
            $params[] = '%' . $search . '%';
        }
        $where = 'WHERE ' . implode(' OR ', $likeClauses);
    }
}

// POST handling
$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($action === 'create' && $table) {
        $bean = R::dispense($table);
        foreach ($_POST as $k => $v) {
            if ($k === 'id') continue;
            $bean[$k] = $v;
        }
        R::store($bean);
        $message = '<span class="text-success">Record created.</span>';
        $action = 'list';
    }
    if ($action === 'update' && $table && isset($_POST['id'])) {
        $bean = R::load($table, (int)$_POST['id']);
        foreach ($_POST as $k => $v) {
            if ($k === 'id') continue;
            $bean[$k] = $v;
        }
        R::store($bean);
        $message = '<span class="text-success">Record updated.</span>';
        $action = 'list';
    }
    if ($action === 'delete' && $table && isset($_POST['id'])) {
        $bean = R::load($table, (int)$_POST['id']);
        R::trash($bean);
        $message = '<span class="text-success">Record deleted.</span>';
        $action = 'list';
    }
}

// Export CSV/Excel
if ($action === 'export' && $table && isset($_GET['type'])) {
    $type = $_GET['type'];
    $filename = $table . '_' . date('Ymd_His');
    $records = R::findAll($table);
    $cols = [];
    if ($records) {
        $cols = array_keys((array)reset($records)->export());
    }
    if ($type === 'csv') {
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '.csv"');
        $out = fopen('php://output', 'w');
        fputcsv($out, $cols);
        foreach ($records as $rec) {
            fputcsv($out, array_map('strval', array_values((array)$rec)));
        }
        fclose($out);
        exit;
    }
    if ($type === 'excel') {
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment; filename="' . $filename . '.xls"');
        echo implode("\t", $cols) . "\n";
        foreach ($records as $rec) {
            echo implode("\t", array_map('strval', array_values((array)$rec))) . "\n";
        }
        exit;
    }
}

// Schema AJAX (for modal)
if ($action === 'schema' && $table) {
    header('Content-Type: text/plain; charset=utf-8');
    $columns = R::getColumns($table);
    foreach ($columns as $col => $type) {
        echo "$col : $type\n";
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title>MwalimuLink™ - Administrator Portal</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<style>
  body { background: #f5f7fa; }
  main { max-width: 980px; margin: 2rem auto; background: white; border-radius: 10px; box-shadow: 0 0 15px #0001; padding: 2rem; }
  .table-responsive { margin-top: 1rem; }
  .actions { gap: 0.5rem; }
</style>
<script>
function showSchemaModal() {
    var table = document.getElementById('tableSelect').value;
    if (!table) return alert('Select a table first.');
    fetch('?table=' + encodeURIComponent(table) + '&action=schema')
      .then(r => r.text())
      .then(data => {
        document.getElementById('schemaText').textContent = data;
        var modal = new bootstrap.Modal(document.getElementById('schemaModal'));
        modal.show();
      });
}
function exportTable(type) {
    var table = document.getElementById('tableSelect').value;
    if (!table) return alert('Select a table first.');
    window.location.href = '?table=' + encodeURIComponent(table) + '&action=export&type=' + type;
}
</script>
</head>
<body>
<main>
<header class="mb-4">
    <h1 class="mb-0">MwalimuLink™ - Administrator Portal</h1>
</header>

<form method="get" class="row g-3 align-items-center mb-4">
    <div class="col-auto">
        <label for="tableSelect" class="col-form-label"><b>Table:</b></label>
    </div>
    <div class="col-auto">
        <select name="table" id="tableSelect" class="form-select" onchange="this.form.submit()" required>
            <option value="">-- Select Table --</option>
            <?php foreach ($tables as $t): ?>
                <option value="<?=h($t)?>" <?= $t === $table ? 'selected' : '' ?>><?=h($t)?></option>
            <?php endforeach; ?>
        </select>
        <input type="hidden" name="action" value="list" />
    </div>
</form>

<?php if ($message): ?>
    <div class="alert alert-success"><?= $message ?></div>
<?php endif; ?>

<?php if ($table === ''): ?>
    <p>Select a table to manage.</p>

<?php elseif ($action === 'list'): ?>
    <h2 class="mt-4">Table: <?=h($table)?></h2>
    <div class="d-flex flex-wrap actions mb-3">
        <a href="?table=<?=h($table)?>&action=create" class="btn btn-primary">Create New Record</a>
        <button type="button" class="btn btn-success" onclick="exportTable('csv')">Export CSV</button>
        <button type="button" class="btn btn-success" onclick="exportTable('excel')">Export Excel</button>
        <button type="button" class="btn btn-outline-secondary" onclick="showSchemaModal()">View Schema</button>
    </div>

    <form method="get" class="row g-2 mb-3">
        <input type="hidden" name="table" value="<?=h($table)?>">
        <input type="hidden" name="action" value="list">
        <div class="col-auto">
            <input type="search" name="search" value="<?=h($search)?>" class="form-control" placeholder="Search in text fields..." />
        </div>
        <div class="col-auto">
            <button type="submit" class="btn btn-primary">Filter</button>
            <?php if ($search !== ''): ?>
                <a href="?table=<?=h($table)?>&action=list" class="btn btn-secondary">Clear</a>
            <?php endif; ?>
        </div>
    </form>

    <?php
    if ($where) {
        $total = R::count($table, $where, $params);
    } else {
        $total = R::count($table);
    }
    $offset = ($page - 1) * $per_page;

    if ($where) {
        $records = R::findAll($table, "$where LIMIT ? OFFSET ?", array_merge($params, [$per_page, $offset]));
    } else {
        $records = R::findAll($table, 'LIMIT ? OFFSET ?', [$per_page, $offset]);
    }

    if ($total === 0) {
        echo "<p>No records found.</p>";
    } else {
        echo '<div class="table-responsive"><table class="table table-striped table-bordered align-middle"><thead><tr>';
        $first = reset($records);
        if ($first) {
            $cols = array_keys((array)$first->export());
            $cols_to_show = array_slice($cols, 0, 3);
            foreach ($cols_to_show as $col) {
                echo '<th>' . h($col) . '</th>';
            }
            echo '<th>Actions</th>';
            echo '</tr></thead><tbody>';
            foreach ($records as $record) {
                echo '<tr>';
                foreach ($cols_to_show as $col) {
                    $val = $record->$col;
                    echo '<td title="' . h($val) . '">' . h($val) . '</td>';
                }
                echo '<td>';
                echo '<a href="?table=' . h($table) . '&action=details&id=' . $record->id . '" class="btn btn-sm btn-info me-1">Details</a>';
                echo '<a href="?table=' . h($table) . '&action=update&id=' . $record->id . '" class="btn btn-sm btn-warning me-1">Edit</a>';
                echo '<a href="?table=' . h($table) . '&action=delete&id=' . $record->id . '" class="btn btn-sm btn-danger">Delete</a>';
                echo '</td>';
                echo '</tr>';
            }
            echo '</tbody></table></div>';

            $base_url = "?table=" . urlencode($table) . "&action=list";
            if ($search !== '') {
                $base_url .= "&search=" . urlencode($search);
            }
            paginate($total, $per_page, $page, $base_url);
        }
    }
    ?>

<?php elseif ($action === 'details' && isset($_GET['id'])): ?>
    <?php
    $id = (int)$_GET['id'];
    $bean = R::load($table, $id);
    if (!$bean->id) {
        echo "<p>Record not found.</p>";
    } else {
        echo "<h2>Details of Record #".h($id)." in ".h($table)."</h2>";
        echo '<table class="table table-bordered">';
        foreach ($bean as $col => $val) {
            echo '<tr><th>' . h($col) . '</th><td>' . nl2br(h($val)) . '</td></tr>';
        }
        echo '</table>';
        echo '<a href="?table=' . h($table) . '&action=list" class="btn btn-secondary">Back to list</a>';
    }
    ?>

<?php elseif ($action === 'create'): ?>
    <h2>Create New Record in <?=h($table)?></h2>
    <form method="post" action="?table=<?=h($table)?>&action=create">
        <?php
        $columns = R::inspect($table);
        foreach ($columns as $col => $type):
            if ($col === 'id') continue;
        ?>
            <div class="mb-3">
                <label for="<?=h($col)?>" class="form-label"><?=h($col)?></label>
                <input type="text" id="<?=h($col)?>" name="<?=h($col)?>" class="form-control" />
            </div>
        <?php endforeach; ?>
        <button type="submit" class="btn btn-primary">Create</button>
        <a href="?table=<?=h($table)?>&action=list" class="btn btn-secondary">Cancel</a>
    </form>

<?php elseif ($action === 'update' && isset($_GET['id'])): ?>
    <?php
    $id = (int)$_GET['id'];
    $bean = R::load($table, $id);
    if (!$bean->id) {
        echo "<p>Record not found.</p>";
    } else {
    ?>
    <h2>Update Record #<?=h($id)?> in <?=h($table)?></h2>
    <form method="post" action="?table=<?=h($table)?>&action=update&id=<?=h($id)?>">
        <input type="hidden" name="id" value="<?=h($id)?>">
        <?php foreach ($bean as $col => $val):
            if ($col === 'id') continue;
        ?>
            <div class="mb-3">
                <label for="<?=h($col)?>" class="form-label"><?=h($col)?></label>
                <input type="text" id="<?=h($col)?>" name="<?=h($col)?>" value="<?=h($val)?>" class="form-control" />
            </div>
        <?php endforeach; ?>
        <button type="submit" class="btn btn-primary">Update</button>
        <a href="?table=<?=h($table)?>&action=list" class="btn btn-secondary">Cancel</a>
    </form>
    <?php } ?>

<?php elseif ($action === 'delete' && isset($_GET['id'])): ?>
    <?php
    $id = (int)$_GET['id'];
    $bean = R::load($table, $id);
    if (!$bean->id) {
        echo "<p>Record not found.</p>";
    } else {
    ?>
    <h2>Delete Record #<?=h($id)?> in <?=h($table)?></h2>
    <form method="post" action="?table=<?=h($table)?>&action=delete&id=<?=h($id)?>">
        <input type="hidden" name="id" value="<?=h($id)?>">
        <div class="alert alert-warning">Are you sure you want to delete this record?</div>
        <button type="submit" class="btn btn-danger">Delete</button>
        <a href="?table=<?=h($table)?>&action=list" class="btn btn-secondary">Cancel</a>
    </form>
    <?php } ?>

<?php else: ?>
    <p>Unknown action.</p>
<?php endif; ?>

<footer class="mt-5 text-center text-muted small">
    MwalimuLink™ Administrator Portal &mdash; Powered by SchoolsNetKenya.com
</footer>
</main>

<!-- Bootstrap Modal for Schema -->
<div class="modal fade" id="schemaModal" tabindex="-1" aria-labelledby="schemaModalLabel" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="schemaModalLabel">Table Schema</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <pre id="schemaText" style="white-space: pre-wrap;"></pre>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>
</body>
</html>
