<?php
require_once 'rb.php';

// --- CONFIGURATION ---
$db_file = 'mwalimu.db';
$dsn = "sqlite:$db_file";
R::setup($dsn);
R::freeze(false);

// --- HELPER ---
function h($str) { return htmlspecialchars($str, ENT_QUOTES, 'UTF-8'); }

// --- ROUTE PARAMS ---
$table = $_GET['table'] ?? '';
$action = $_GET['action'] ?? 'list';
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 10;
$search = trim($_GET['search'] ?? '');

// --- Fetch tables ---
$tables = R::inspect();

if ($table !== '' && !in_array($table, $tables)) {
    die('Invalid table.');
}

// --- Pagination ---
function paginate($total, $per_page, $page, $base_url) {
    $pages = max(1, ceil($total / $per_page));
    $page = max(1, min($page, $pages));
    echo '<nav aria-label="Pagination" style="margin-top:1rem;"><ul class="pagination" style="display:flex;gap:0.5rem;list-style:none;padding:0;">';
    echo $page > 1
        ? '<li><a href="' . h($base_url) . '&page=1">First</a></li>'
        : '<li><span style="opacity:0.5;">First</span></li>';
    echo $page > 1
        ? '<li><a href="' . h($base_url) . '&page=' . ($page - 1) . '">Prev</a></li>'
        : '<li><span style="opacity:0.5;">Prev</span></li>';
    echo '<li><span>Page ' . $page . ' of ' . $pages . ' (Total: ' . $total . ')</span></li>';
    echo $page < $pages
        ? '<li><a href="' . h($base_url) . '&page=' . ($page + 1) . '">Next</a></li>'
        : '<li><span style="opacity:0.5;">Next</span></li>';
    echo $page < $pages
        ? '<li><a href="' . h($base_url) . '&page=' . $pages . '">Last</a></li>'
        : '<li><span style="opacity:0.5;">Last</span></li>';
    echo '</ul></nav>';
}

// --- Search filter ---
$where = '';
$params = [];
if ($search !== '' && $table !== '') {
    $columns = R::getColumns($table);
    $textColumns = [];
    foreach ($columns as $col => $type) {
        if (stripos($type, 'char') !== false || stripos($type, 'text') !== false) {
            $textColumns[] = $col;
        }
    }
    if ($textColumns) {
        $likeClauses = [];
        foreach ($textColumns as $col) {
            $likeClauses[] = "$col LIKE ?";
            $params[] = '%' . $search . '%';
        }
        $where = 'WHERE ' . implode(' OR ', $likeClauses);
    }
}

// --- POST handling ---
$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($action === 'create' && $table) {
        $bean = R::dispense($table);
        foreach ($_POST as $k => $v) {
            if ($k === 'id') continue;
            $bean[$k] = $v;
        }
        R::store($bean);
        $message = '<b style="color:green">Record created.</b>';
        $action = 'list';
    }
    if ($action === 'update' && $table && isset($_POST['id'])) {
        $bean = R::load($table, (int)$_POST['id']);
        foreach ($_POST as $k => $v) {
            if ($k === 'id') continue;
            $bean[$k] = $v;
        }
        R::store($bean);
        $message = '<b style="color:green">Record updated.</b>';
        $action = 'list';
    }
    if ($action === 'delete' && $table && isset($_POST['id'])) {
        $bean = R::load($table, (int)$_POST['id']);
        R::trash($bean);
        $message = '<b style="color:green">Record deleted.</b>';
        $action = 'list';
    }
}

// --- Export CSV/Excel ---
if ($action === 'export' && $table && isset($_GET['type'])) {
    $type = $_GET['type'];
    $filename = $table . '_' . date('Ymd_His');
    $records = R::findAll($table);
    $cols = [];
    if ($records) {
        $cols = array_keys((array)reset($records)->export());
    }
    if ($type === 'csv') {
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '.csv"');
        $out = fopen('php://output', 'w');
        fputcsv($out, $cols);
        foreach ($records as $rec) {
            fputcsv($out, array_map('strval', array_values((array)$rec)));
        }
        fclose($out);
        exit;
    }
    if ($type === 'excel') {
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment; filename="' . $filename . '.xls"');
        echo implode("\t", $cols) . "\n";
        foreach ($records as $rec) {
            echo implode("\t", array_map('strval', array_values((array)$rec))) . "\n";
        }
        exit;
    }
}

// --- Schema AJAX (for modal) ---
if ($action === 'schema' && $table) {
    header('Content-Type: text/plain; charset=utf-8');
    $columns = R::getColumns($table);
    foreach ($columns as $col => $type) {
        echo "$col : $type\n";
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title>MwalimuLink™ - Administrator Portal</title>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@picocss/pico@1.4.0/css/pico.min.css" />
<style>
  body {
    background: #f5f7fa;
  }
  main {
    max-width: 960px;
    margin: 2rem auto;
    background: white;
    border-radius: 8px;
    box-shadow: 0 0 15px rgba(0,0,0,0.08);
    padding: 2rem;
  }
  .actions {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
    margin-bottom: 1rem;
  }
  .pagination {
    justify-content: center;
  }
  .modal {
    display:none;
    position:fixed; top:0; left:0; width:100vw; height:100vh;
    background:rgba(0,0,0,0.4); z-index:1000;
    align-items:center; justify-content:center;
  }
  .modal-content {
    background:white; padding:2rem; border-radius:8px; max-width:90vw; max-height:80vh; overflow:auto;
  }
</style>
<script>
function showSchemaModal() {
    var table = document.getElementById('tableSelect').value;
    if (!table) return alert('Select a table first.');
    fetch('?table=' + encodeURIComponent(table) + '&action=schema')
      .then(r => r.text())
      .then(data => {
        document.getElementById('schemaText').textContent = data;
        document.getElementById('schemaModal').style.display = 'flex';
      });
}
function closeSchemaModal() {
    document.getElementById('schemaModal').style.display = 'none';
}
function exportTable(type) {
    var table = document.getElementById('tableSelect').value;
    if (!table) return alert('Select a table first.');
    window.location.href = '?table=' + encodeURIComponent(table) + '&action=export&type=' + type;
}
</script>
</head>
<body>
<main>
<header>
    <h1>MwalimuLink™ - Administrator Portal</h1>
</header>

<section>
    <form method="get" style="display:flex; gap:1rem; align-items:center; flex-wrap:wrap;">
        <label for="tableSelect"><b>Table:</b></label>
        <select name="table" id="tableSelect" onchange="this.form.submit()" required>
            <option value="">-- Select Table --</option>
            <?php foreach ($tables as $t): ?>
                <option value="<?=h($t)?>" <?= $t === $table ? 'selected' : '' ?>><?=h($t)?></option>
            <?php endforeach; ?>
        </select>
        <input type="hidden" name="action" value="list" />
    </form>
</section>

<section>
    <?php if ($message): ?>
        <article style="color:green;"><?= $message ?></article>
    <?php endif; ?>

    <?php if ($table === ''): ?>
        <p>Select a table to manage.</p>

    <?php elseif ($action === 'list'): ?>
        <h2>Table: <?=h($table)?></h2>
        <div class="actions">
            <a href="?table=<?=h($table)?>&action=create" role="button" class="secondary">Create New Record</a>
            <button type="button" class="secondary" onclick="exportTable('csv')">Export CSV</button>
            <button type="button" class="secondary" onclick="exportTable('excel')">Export Excel</button>
            <button type="button" class="contrast outline" onclick="showSchemaModal()">View Schema</button>
        </div>

        <form method="get" class="search-form" style="margin-bottom:1em;">
            <input type="hidden" name="table" value="<?=h($table)?>">
            <input type="hidden" name="action" value="list">
            <input type="search" name="search" value="<?=h($search)?>" placeholder="Search in text fields..." />
            <button type="submit">Filter</button>
            <?php if ($search !== ''): ?>
                <a href="?table=<?=h($table)?>&action=list" role="button" class="secondary">Clear</a>
            <?php endif; ?>
        </form>

        <?php
        if ($where) {
            $total = R::count($table, $where, $params);
        } else {
            $total = R::count($table);
        }
        $offset = ($page - 1) * $per_page;

        if ($where) {
            $records = R::findAll($table, "$where LIMIT ? OFFSET ?", array_merge($params, [$per_page, $offset]));
        } else {
            $records = R::findAll($table, 'LIMIT ? OFFSET ?', [$per_page, $offset]);
        }

        if ($total === 0) {
            echo "<p>No records found.</p>";
        } else {
            echo '<table><thead><tr>';
            $first = reset($records);
            if ($first) {
                $cols = array_keys((array)$first->export());
                $cols_to_show = array_slice($cols, 0, 3);
                foreach ($cols_to_show as $col) {
                    echo '<th>' . h($col) . '</th>';
                }
                echo '<th>Actions</th>';
                echo '</tr></thead><tbody>';
                foreach ($records as $record) {
                    echo '<tr>';
                    foreach ($cols_to_show as $col) {
                        $val = $record->$col;
                        echo '<td title="' . h($val) . '">' . h($val) . '</td>';
                    }
                    echo '<td>';
                    echo '<a href="?table=' . h($table) . '&action=details&id=' . $record->id . '">Details</a> | ';
                    echo '<a href="?table=' . h($table) . '&action=update&id=' . $record->id . '">Edit</a> | ';
                    echo '<a href="?table=' . h($table) . '&action=delete&id=' . $record->id . '">Delete</a>';
                    echo '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';

                $base_url = "?table=" . urlencode($table) . "&action=list";
                if ($search !== '') {
                    $base_url .= "&search=" . urlencode($search);
                }
                paginate($total, $per_page, $page, $base_url);
            }
        }
        ?>

    <?php elseif ($action === 'details' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
            echo "<h2>Details of Record #".h($id)." in ".h($table)."</h2>";
            echo '<table>';
            foreach ($bean as $col => $val) {
                echo '<tr><th>' . h($col) . '</th><td>' . nl2br(h($val)) . '</td></tr>';
            }
            echo '</table>';
            echo '<p><a href="?table=' . h($table) . '&action=list" role="button" class="secondary">Back to list</a></p>';
        }
        ?>

    <?php elseif ($action === 'create'): ?>
        <h2>Create New Record in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=create">
            <?php
            $columns = R::inspect($table);
            foreach ($columns as $col => $type):
                if ($col === 'id') continue;
            ?>
                <label for="<?=h($col)?>"><?=h($col)?></label>
                <input type="text" id="<?=h($col)?>" name="<?=h($col)?>" />
            <?php endforeach; ?>
            <button type="submit">Create</button>
            <a href="?table=<?=h($table)?>&action=list" role="button" class="secondary">Cancel</a>
        </form>

    <?php elseif ($action === 'update' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
        ?>
        <h2>Update Record #<?=h($id)?> in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=update&id=<?=h($id)?>">
            <input type="hidden" name="id" value="<?=h($id)?>">
            <?php foreach ($bean as $col => $val):
                if ($col === 'id') continue;
            ?>
                <label for="<?=h($col)?>"><?=h($col)?></label>
                <input type="text" id="<?=h($col)?>" name="<?=h($col)?>" value="<?=h($val)?>" />
            <?php endforeach; ?>
            <button type="submit">Update</button>
            <a href="?table=<?=h($table)?>&action=list" role="button" class="secondary">Cancel</a>
        </form>
        <?php } ?>

    <?php elseif ($action === 'delete' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
        ?>
        <h2>Delete Record #<?=h($id)?> in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=delete&id=<?=h($id)?>">
            <input type="hidden" name="id" value="<?=h($id)?>">
            <p>Are you sure you want to delete this record?</p>
            <button type="submit" class="contrast">Delete</button>
            <a href="?table=<?=h($table)?>&action=list" role="button" class="secondary">Cancel</a>
        </form>
        <?php } ?>

    <?php else: ?>
        <p>Unknown action.</p>
    <?php endif; ?>
</section>

<footer>
    <p style="text-align:center; font-size:small; color:#666;">MwalimuLink™ Administrator Portal &mdash; Powered by SchoolsNetKenya.com</p>
</footer>
</main>

<!-- Schema Modal -->
<div class="modal" id="schemaModal" onclick="if(event.target==this)closeSchemaModal()">
  <div class="modal-content">
    <h3>Table Schema</h3>
    <pre id="schemaText" style="white-space:pre-wrap;max-height:400px;"></pre>
    <button onclick="closeSchemaModal()">Close</button>
  </div>
</div>
</body>
</html>
