<?php
// staff.php - Final version with search, pagination, and CRUD

require_once 'rb.php';

// --- CONFIGURATION ---
$db_file = 'mwalimu.db'; // Adjust your SQLite DB or DSN here
$dsn = "sqlite:$db_file";
R::setup($dsn);
R::freeze(false);

// --- HELPER ---
function h($str) { return htmlspecialchars($str, ENT_QUOTES, 'UTF-8'); }

// --- ROUTE PARAMS ---
$table = $_GET['table'] ?? '';
$action = $_GET['action'] ?? 'list';
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 10;
$search = trim($_GET['search'] ?? '');

// --- Fetch tables ---
$tables = R::inspect();

if ($table !== '' && !in_array($table, $tables)) {
    die('Invalid table.');
}

// --- Pagination ---
function paginate($total, $per_page, $page, $base_url) {
    $pages = max(1, ceil($total / $per_page));
    $page = max(1, min($page, $pages));
    echo '<nav aria-label="Pagination" style="margin-top:1rem;">';
    echo '<ul class="pagination" style="list-style:none; display:flex; gap:0.5rem; padding:0;">';

    echo $page > 1
        ? '<li><a href="' . h($base_url) . '&page=1">First</a></li>'
        : '<li style="color:#999; padding:0.3rem 0.6rem;">First</li>';

    echo $page > 1
        ? '<li><a href="' . h($base_url) . '&page=' . ($page - 1) . '">Prev</a></li>'
        : '<li style="color:#999; padding:0.3rem 0.6rem;">Prev</li>';

    echo '<li style="padding:0.3rem 0.6rem;">page ' . $page . ' of ' . $pages . ' (Total Recs: ' . $total . ')</li>';

    echo $page < $pages
        ? '<li><a href="' . h($base_url) . '&page=' . ($page + 1) . '">Next</a></li>'
        : '<li style="color:#999; padding:0.3rem 0.6rem;">Next</li>';

    echo $page < $pages
        ? '<li><a href="' . h($base_url) . '&page=' . $pages . '">Last</a></li>'
        : '<li style="color:#999; padding:0.3rem 0.6rem;">Last</li>';

    echo '</ul></nav>';
}

// --- Search filter ---
$where = '';
$params = [];
if ($search !== '' && $table !== '') {
    $columns = R::getColumns($table);
    $textColumns = [];
    foreach ($columns as $col => $type) {
        if (stripos($type, 'char') !== false || stripos($type, 'text') !== false) {
            $textColumns[] = $col;
        }
    }
    if ($textColumns) {
        $likeClauses = [];
        foreach ($textColumns as $col) {
            $likeClauses[] = "$col LIKE ?";
            $params[] = '%' . $search . '%';
        }
        $where = 'WHERE ' . implode(' OR ', $likeClauses);
    }
}

// --- POST handling ---
$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($action === 'create' && $table) {
        $bean = R::dispense($table);
        foreach ($_POST as $k => $v) {
            if ($k === 'id') continue;
            $bean[$k] = $v;
        }
        R::store($bean);
        $message = '<b style="color:green">Record created.</b>';
        $action = 'list';
    }
    if ($action === 'update' && $table && isset($_POST['id'])) {
        $bean = R::load($table, (int)$_POST['id']);
        foreach ($_POST as $k => $v) {
            if ($k === 'id') continue;
            $bean[$k] = $v;
        }
        R::store($bean);
        $message = '<b style="color:green">Record updated.</b>';
        $action = 'list';
    }
    if ($action === 'delete' && $table && isset($_POST['id'])) {
        $bean = R::load($table, (int)$_POST['id']);
        R::trash($bean);
        $message = '<b style="color:green">Record deleted.</b>';
        $action = 'list';
    }
}

?><!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title>MwalimuLink™ - Staff Portal</title>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@picocss/pico@1.4.0/css/pico.min.css" />
<script src="https://cdn.jsdelivr.net/npm/prototype@1.7.3/dist/prototype.min.js"></script>
<style>
  body {
    display: flex;
    justify-content: center;
    align-items: flex-start;
    min-height: 100vh;
    padding: 2rem;
    background: #f5f7fa;
  }
  main.card {
    max-width: 960px;
    width: 100%;
    padding: 2rem;
    background: white;
    border-radius: 8px;
    box-shadow: 0 0 15px rgba(0,0,0,0.1);
  }
  table {
    width: 100%;
    border-collapse: collapse;
  }
  th, td {
    padding: 0.3rem 0.5rem;
    border: 1px solid #ccc;
    text-align: left;
    max-width: 200px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
  }
  nav.pagination ul {
    list-style: none;
    padding: 0;
    display: flex;
    gap: 0.5rem;
  }
  nav.pagination li a {
    padding: 0.3rem 0.6rem;
    border: 1px solid #ccc;
    border-radius: 4px;
    text-decoration: none;
  }
  nav.pagination li a[aria-current="page"] {
    background: #007bff;
    color: white;
    pointer-events: none;
  }
  nav > a {
    margin-right: 0.5rem;
  }
  form.search-form {
    margin-bottom: 1rem;
  }
</style>
</head>
<body>
<main class="card" role="main">
<header><img src="mwalimu-link-logo-clean.png" alt="Logo" class="logo">
    <h1>Staff Portal</h1>
</header>
<div>
<section>
    <h2>MwalimuLink™ </h2>
    <!-- NAVIGATOR <nav>
        <?php foreach ($tables as $t): ?>
            <a href="?table=<?=h($t)?>&action=list"><?=h($t)?></a>
        <?php endforeach; ?>
    </nav>-->
    <nav style="margin-bottom: 1rem;">
  <label for="table-select" style="font-weight: 600; margin-right: 0.5rem;">Select:</label>
  <select id="table-select" style="padding: 0.25em 0.5em; font-size: 1rem; border-radius: 4px; border: 1px solid #ccc;">
    <option value="">-- Choose a table --</option>
    <?php foreach ($tables as $t): ?>
      <option value="?table=<?=h($t)?>&action=list" <?= ($table === $t) ? 'selected' : '' ?>>
        <?=h($t)?>
      </option>
    <?php endforeach; ?>
  </select>
</nav>

<script>
  document.getElementById('table-select').addEventListener('change', function() {
    if (this.value) {
      window.location.href = this.value;
    }
  });
</script>
<!-- NAVIGATOR END -->
</section>

<section>
    <?php if ($message): ?>
        <p><?= $message ?></p>
    <?php endif; ?>

    <?php if ($table === ''): ?>
        <p>Select a table to manage.</p>

    <?php elseif ($action === 'list'): ?>
        <h2>Table: <?=h($table)?></h2>
        <nav>
            <!--<a href="?table=<?=h($table)?>&action=create">Create New Record --></a>
        </nav>

        <form method="get" class="search-form" action="">
            <input type="hidden" name="table" value="<?=h($table)?>">
            <input type="hidden" name="action" value="list">
            <label for="search">Search:</label>
            <input type="search" id="search" name="search" value="<?=h($search)?>" placeholder="Search in text fields..." />
            <button type="submit">Filter</button>
            <?php if ($search !== ''): ?>
                <a href="?table=<?=h($table)?>&action=list">Clear</a>
            <?php endif; ?>
        </form>

        <?php
        if ($where) {
            $total = R::count($table, $where, $params);
        } else {
            $total = R::count($table);
        }
        $offset = ($page - 1) * $per_page;

        if ($where) {
            $records = R::findAll($table, "$where LIMIT ? OFFSET ?", array_merge($params, [$per_page, $offset]));
        } else {
            $records = R::findAll($table, 'LIMIT ? OFFSET ?', [$per_page, $offset]);
        }

        if ($total === 0) {
            echo "<p>No records found.</p>";
        } else {
            echo '<table><thead><tr>';
            $first = reset($records);
            if ($first) {
                $cols = array_keys((array)$first->export());
                $cols_to_show = array_slice($cols, 0, 3);
                foreach ($cols_to_show as $col) {
                    echo '<th>' . h($col) . '</th>';
                }
                echo '<th>Actions</th>';
                echo '</tr></thead><tbody>';
                foreach ($records as $record) {
                    echo '<tr>';
                    foreach ($cols_to_show as $col) {
                        $val = $record->$col;
                        echo '<td title="' . h($val) . '">' . h($val) . '</td>';
                    }
                    echo '<td>';
                    echo '<a href="?table=' . h($table) . '&action=details&id=' . $record->id . '">Details</a> | ';
                    echo '<!-- <a href="?table=' . h($table) . '&action=update&id=' . $record->id . '">Edit</a> | ';
                    echo '<a href="?table=' . h($table) . '&action=delete&id=' . $record->id . '">Delete</a>-->';
                    echo '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';

                $base_url = "?table=" . urlencode($table) . "&action=list";
                if ($search !== '') {
                    $base_url .= "&search=" . urlencode($search);
                }
                paginate($total, $per_page, $page, $base_url);
            }
        }
        ?>

    <?php elseif ($action === 'details' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
            echo "<h2>Details of Record #".h($id)." in ".h($table)."</h2>";
            echo '<table>';
            foreach ($bean as $col => $val) {
                echo '<tr><th>' . h($col) . '</th><td>' . nl2br(h($val)) . '</td></tr>';
            }
            echo '</table>';
            echo '<p><a href="?table=' . h($table) . '&action=list">Back to list</a></p>';
        }
        ?>

    <?php elseif ($action === 'create'): ?>
        <h2>Create New Record in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=create">
            <?php
            $columns = R::inspect($table);
            foreach ($columns as $col => $type):
                if ($col === 'id') continue;
            ?>
                <label for="<?=h($col)?>"><?=h($col)?></label>
                <input type="text" id="<?=h($col)?>" name="<?=h($col)?>" />
            <?php endforeach; ?>
            <button type="submit">Create</button>
            <a href="?table=<?=h($table)?>&action=list">Cancel</a>
        </form>

    <?php elseif ($action === 'update' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
        ?>
        <h2>Update Record #<?=h($id)?> in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=update&id=<?=h($id)?>">
            <input type="hidden" name="id" value="<?=h($id)?>">
            <?php
            foreach ($bean as $col => $val):
                if ($col === 'id') continue;
            ?>
                <label for="<?=h($col)?>"><?=h($col)?></label>
                <input type="text" id="<?=h($col)?>" name="<?=h($col)?>" value="<?=h($val)?>" />
            <?php endforeach; ?>
            <button type="submit">Update</button>
            <a href="?table=<?=h($table)?>&action=list">Cancel</a>
        </form>
        <?php } ?>

    <?php elseif ($action === 'delete' && isset($_GET['id'])): ?>
        <?php
        $id = (int)$_GET['id'];
        $bean = R::load($table, $id);
        if (!$bean->id) {
            echo "<p>Record not found.</p>";
        } else {
        ?>
        <h2>Delete Record #<?=h($id)?> in <?=h($table)?></h2>
        <form method="post" action="?table=<?=h($table)?>&action=delete&id=<?=h($id)?>">
            <input type="hidden" name="id" value="<?=h($id)?>">
            <p>Are you sure you want to delete this record?</p>
            <button type="submit">Delete</button>
            <a href="?table=<?=h($table)?>&action=list">Cancel</a>
        </form>
        <?php } ?>

    <?php else: ?>
        <p>Unknown action.</p>
    <?php endif; ?>
</section>

<footer>
    <p style="text-align:center; font-size:small; color:#666;">MwalimuLink™ Staff Portal &mdash; Powered by SchoolsNetKenya.com</p>
</footer>
</main>
</body>
</html>
